import json

from dataclasses import dataclass, asdict, field
from typing import List, Optional

from common import constants as const


@dataclass
class NsxManagerDetails:
    username: str
    ip: str
    port: Optional[int] = None

    def __post_init__(self):
        if not isinstance(self.username, str) or not self.username:
            raise ValueError("Invalid 'nsx_manager_details.username': must be a non-empty string.")

        if not isinstance(self.ip, str) or not self.ip:
            raise ValueError("Invalid 'nsx_manager_details.ip': must be a non-empty string.")

        if self.port:
            if not isinstance(self.port, int):  # Check to allow int like strings. Ex: "8080"
                try:
                    self.port = int(self.port)
                except ValueError as ex:
                    raise ValueError("'nsx_manager_details.port' must be an integer.") from ex


@dataclass
class VcenterDetails:
    ip: str
    username: str

    def __post_init__(self):
        if not isinstance(self.ip, str) or not self.ip:
            raise ValueError("Invalid 'vcenter_details.ip': must be a non-empty string.")

        if not isinstance(self.username, str) or not self.username:
            raise ValueError("Invalid 'vcenter_details.username': must be a non-empty string.")


@dataclass
class ClusterEntry:
    vcenter_cluster_name: str
    tnp_id_to_apply: str
    enter_mm_timeout_minutes: Optional[int] = field(default=None)
    reset_high_performance_params: bool = field(default=False)

    def __post_init__(self):
        if not isinstance(self.vcenter_cluster_name, str) or not self.vcenter_cluster_name:
            raise ValueError("'cluster_entry.vcenter_cluster_name' must be a non-empty string.")

        if not isinstance(self.tnp_id_to_apply, str) or not self.tnp_id_to_apply:
            raise ValueError("Invalid 'cluster_entry.tnp_id_to_apply': must be a non-empty string.")

        if self.enter_mm_timeout_minutes:
            if not isinstance(self.enter_mm_timeout_minutes, int):
                raise ValueError("'cluster_entry.enter_mm_timeout_minutes' must be an integer.")
            if not (const.ENTER_MM_TIMEOUT_MIN_WAIT_IN_MINUTES <= self.enter_mm_timeout_minutes):
                raise ValueError("'cluster_entry.enter_mm_timeout_minutes' must be greater than or"
                                 " equal to %s" % const.ENTER_MM_TIMEOUT_MIN_WAIT_IN_MINUTES)
        else:
            self.enter_mm_timeout_minutes = const.ENTER_MM_TIMEOUT_DEFAULT_VALUE_IN_MINUTES


@dataclass
class Config:
    nsx_manager_details: NsxManagerDetails
    vcenter_details: VcenterDetails
    cluster_entry_list: List[ClusterEntry]

    def __post_init__(self):
        self.nsx_manager_details.__post_init__()
        self.vcenter_details.__post_init__()

        if not isinstance(self.cluster_entry_list, list):
            raise ValueError("Invalid 'cluster_entry_list': must be a list.")
        for cluster_entry in self.cluster_entry_list:
            if not isinstance(cluster_entry, ClusterEntry):
                raise ValueError("Invalid entry in 'cluster_entry_list': must be a ClusterEntry.")
            cluster_entry.__post_init__()

    def pretty_print(self) -> str:
        return json.dumps(asdict(self), indent=4)
