import enum
import json

from typing import List


class OpStatus(enum.Enum):
    SUCCESS = "SUCCESS"
    FAILED = "FAILED"


class TnProcessingInfo:
    def __init__(self, status: OpStatus, error_message: str, pause_after_failure: bool = False):
        self.status = status
        self.error_message = error_message
        self.pause_after_failure = pause_after_failure

    def to_dict(self):
        return {
            "status": self.status.value,
            "error_message": self.error_message,
            "pause_after_failure": self.pause_after_failure
        }

    def pretty_print(self) -> str:
        return json.dumps(self.to_dict(), indent=4)


class HostSwitch:
    def __init__(self,
                 id: str,
                 name: str,
                 mode: str,
                 spec):
        self.id = id
        self.name = name
        self.mode = mode
        self.spec_json = spec

    def to_dict(self):
        return {
            "id": self.id,
            "name": self.name,
            "mode": self.mode
        }

    @staticmethod
    def pretty_print_list(host_switch_list: List['HostSwitch']) -> str:
        return json.dumps([host_switch.to_dict() for host_switch in host_switch_list], indent=4)


class HostTnModel:
    def __init__(self,
                 id: str,
                 host_mo_id: str,
                 host_switch_list: List[HostSwitch],
                 tn_api_resp_json):
        self.id = id
        self.host_mo_id = host_mo_id
        self.host_switch_list = host_switch_list
        self.tn_api_resp_json = tn_api_resp_json

    def to_dict(self):
        return {
            "id": self.id,
            "host_mo_id": self.host_mo_id,
            "host_switch_list": [switch.to_dict() for switch in self.host_switch_list]
        }

    def pretty_print(self) -> str:
        return json.dumps(self.to_dict(), indent=4)

    @staticmethod
    def pretty_print_list(tn_list: List['HostTnModel']) -> str:
        return json.dumps([tn.to_dict() for tn in tn_list], indent=4)


class TncModel:
    def __init__(self,
                 id: str,
                 resource_type: str,
                 parent_compute_collection_id: str,
                 cluster_id: str,
                 transport_node_profile_path: str,
                 tnc_api_resp_json):
        self.id = id
        self.resource_type = resource_type
        self.parent_compute_collection_id = parent_compute_collection_id
        self.cluster_id = cluster_id
        self.transport_node_profile_path = transport_node_profile_path
        self.tnc_api_resp_json = tnc_api_resp_json

    def to_dict(self):
        return {
            "id": self.id,
            "resource_type": self.resource_type,
            "parent_compute_collection_id": self.parent_compute_collection_id,
            "cluster_id": self.cluster_id,
            "transport_node_profile_path": self.transport_node_profile_path,
        }

    def pretty_print(self) -> str:
        return json.dumps(self.to_dict(), indent=4)
