import argparse
import getpass
import json
import os
import sys
import ipaddress
import socket

from typing import Tuple

from clients.nsx_client import NSXClient
from clients.nsx_utils import NsxUtils
from clients.vc_client import VCClient
from clients.vc_utils import VcUtils
from common.logger import setup_logger
from model.config import VcenterDetails, Config, NsxManagerDetails, ClusterEntry
from service.enable_uens_service import UensEnablement

logger = setup_logger()


def read_config_file(config_file_path: str) -> Config:
    try:
        if not os.path.exists(config_file_path):
            raise FileNotFoundError("Config file NOT found at : %s" % (config_file_path))

        with open(config_file_path, 'r', encoding='utf-8') as f:
            config_data = json.load(f)

        nsx_manager_details = NsxManagerDetails(**config_data["nsx_manager_details"])
        vcenter_details = VcenterDetails(**config_data["vcenter_details"])
        cluster_entry_list = [ClusterEntry(**ce) for ce in config_data["cluster_entry_list"]]

        config = Config(
            nsx_manager_details=nsx_manager_details,
            vcenter_details=vcenter_details,
            cluster_entry_list=cluster_entry_list,
        )

        return config

    except FileNotFoundError as e:
        logger.error(str(e))
    except json.JSONDecodeError as e:
        logger.error("Invalid JSON format when reading config file : %s", str(e))
    except ValueError as e:
        logger.error("Json schema validation failed when reading config file: %s", str(e))
    except Exception as e:
        logger.error("An unexpected error occurred when reading config file : %s", str(e))

    return None


def get_non_empty_password(prompt: str) -> str:
    while True:
        try:
            password = getpass.getpass(prompt)
            if password:
                return password
            else:
                print("Password cannot be empty. Please try again.")
        except KeyboardInterrupt:
            print("\nPassword input cancelled.")
            sys.exit(1)

def get_ips_by_fqdn(fqdn: str):
    try:
        addr_infos = socket.getaddrinfo(fqdn, None, proto=socket.IPPROTO_TCP)
    except socket.gaierror as ex:
        raise Exception("Failed to resolve FQDN %s: %s." % (fqdn, str(ex))) from ex
    ips = set()
    for info in addr_infos:
        family, sockaddr = info[0], info[4]
        if sockaddr:
            if family == socket.AF_INET and sockaddr[0]:
                ips.add(sockaddr[0])
            elif family == socket.AF_INET6 and sockaddr[0] and \
                not ipaddress.ip_address(sockaddr[0]).is_link_local:
                ips.add(sockaddr[0])
    return ips

def get_nsx_connection(config: Config, nsx_password: str, verify_ssl) -> NsxUtils:
    nsx_client = NSXClient(fqdn=config.nsx_manager_details.ip,
                           port=config.nsx_manager_details.port,
                           username=config.nsx_manager_details.username,
                           password=nsx_password, verify_ssl=verify_ssl)
    nsx_utils = NsxUtils(nsx_client=nsx_client)

    version_json = nsx_utils.get_version()
    logger.info("NSX manager connection validated successfully. NSX version : %s",
                version_json["node_version"])

    return nsx_utils


def get_vc_connection(config: Config, vc_password: str, nsx_utils: NsxUtils,
                      verify_ssl) -> Tuple[VcUtils, str]:
    vc_ips = get_ips_by_fqdn(config.vcenter_details.ip)
    vc_ips.add(config.vcenter_details.ip)
    for vc_ip in vc_ips:
        cm_json = nsx_utils.get_cm_json_by_server(server=vc_ip)
        if cm_json and cm_json.get("server"):
            break
    if not cm_json:
        raise Exception("Could NOT find a NSX compute manager with vCenter FQDN %s" % (
            config.vcenter_details.ip))

    cm_ip: str = cm_json["server"]
    cm_port: str = cm_json["reverse_proxy_https_port"]
    cm_id: str = cm_json["id"]

    if not cm_id:
        raise Exception("Could NOT find a NSX compute manager with vCenter FQDN %s" % (
            config.vcenter_details.ip))

    logger.info("For input vCenter FQDN %s found a matching NSX compute manager with Id %s and "
                "server IP %s", config.vcenter_details.ip, cm_id, cm_ip)

    vc_client = VCClient(fqdn=config.vcenter_details.ip,
                         port=cm_port,
                         username=config.vcenter_details.username,
                         password=vc_password)
    vc_utils = VcUtils(vc_client=vc_client, verify_ssl=verify_ssl)

    vc_utils.connect_to_vc()

    return vc_utils, cm_id


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("-f", dest="config_file_path", required=True,
                        help="Absolute path of config file.")
    parser.add_argument('-s', '--skip-ssl-verification',
                        action='store_true', help="Skip TLS/SSL verification.")
    args = parser.parse_args()

    config_file_path: str = args.config_file_path
    logger.info("Config file path provided to script is : %s", config_file_path)
    if not config_file_path:
        raise ValueError("Config file path argument is missing. It is mandatory to provide "
                         "config file absolute path.")

    config: Config = read_config_file(config_file_path=config_file_path)
    if config is None:
        raise Exception("Error while reading config file. Please fix the error and re-try.")
    logger.info("Config data to use will be : \n%s", config.pretty_print())

    # Get NSX password from environment variable or prompt
    nsx_password_env = os.getenv("ENABLE_UENS_SCRIPT_NSX_PASSWORD")
    if nsx_password_env:
        nsx_password = nsx_password_env
        logger.info("NSX password read from environment variable.")
    else:
        nsx_password = get_non_empty_password("Enter NSX password: ")

    # Get vCenter password from environment variable or prompt
    vc_password_env = os.getenv("ENABLE_UENS_SCRIPT_VC_PASSWORD")
    if vc_password_env:
        vc_password = vc_password_env
        logger.info("vCenter password read from environment variable.")
    else:
        vc_password = get_non_empty_password("Enter vCenter user password: ")

    nsx_utils: NsxUtils = get_nsx_connection(config=config, nsx_password=nsx_password,
                                             verify_ssl=not args.skip_ssl_verification)

    vc_utils: VcUtils
    cm_id: str
    vc_utils, cm_id = get_vc_connection(config=config, vc_password=vc_password, nsx_utils=nsx_utils,
                                        verify_ssl=not args.skip_ssl_verification)

    uens_enablement = UensEnablement(config=config,
                                     cm_id=cm_id,
                                     nsx_utils=nsx_utils,
                                     vc_utils=vc_utils)
    uens_enablement.run()


if __name__ == "__main__":
    main()
