import base64
import time
import json
import requests
from six.moves.urllib import parse as urlparse

from common import constants as const
from common.logger import setup_logger

logger = setup_logger()


class DetailedHttpError(requests.exceptions.HTTPError):
    def __init__(self, response):
        self.err_code = 0
        reason = response.text.replace("\n", "")
        try:
            reason_json = response.json()
            if 'error_message' in reason_json:
                reason = reason_json['error_message']
            if 'error_code' in reason_json:
                self.err_code = reason_json['error_code']
        except Exception:
            pass

        http_error_msg = '%s: %s for url: %s' % (response.status_code, reason, response.url)
        super().__init__(http_error_msg, response=response)


class NSXClient:
    ALLOWED_GET_STATUS = [requests.codes.OK]
    ALLOWED_PUT_STATUS = [requests.codes.OK,
                          requests.codes.CREATED,
                          requests.codes.ACCEPTED]

    NULL_CURSOR_PREFIX = '0000'

    def __init__(self, fqdn: str, port: int, username: str, password: str, verify_ssl=True):
        self.fqdn = fqdn
        self.port = port
        if self.port:
            self.base_url = "https://%s:%s" % (self.fqdn, self.port)
        else:
            self.base_url = "https://%s" % (self.fqdn)
        self.username = username
        self.password = password
        self.verify_ssl = verify_ssl
        self.timeout = const.NSX_API_TIMEOUT
        self.content_type = "application/json"
        self.accept_type = "application/json"
        self.session_headers = None
        self.set_session_headers()

    def set_session_headers(self, content: str = None, accept: str = None):
        content_type = self.content_type if content is None else content
        accept_type = self.accept_type if accept is None else accept
        headers = {}

        auth_cred = self.username + ":" + self.password
        auth = base64.b64encode(auth_cred.encode())
        headers['Authorization'] = "Basic %s" % auth.decode()

        headers['Content-Type'] = content_type
        headers['Accept'] = accept_type
        # In case manager API forces to PUT policy created object.
        headers['X-Allow-Overwrite'] = 'true'

        self.session_headers = headers

    def _construct_url(self, endpoint: str):
        url = "%s/%s" % (self.base_url, endpoint)
        return url

    def _rest_call(self, method: str, endpoint: str, data, allowed_status_codes, params=None,
                   raise_ex=True):
        url = self._construct_url(endpoint)
        logger.debug("API : REQUEST method=%s, url=%s, params=%s, data=%s", method, url,
                     str(params), str(data))
        retries = const.NSX_API_RETRIES
        retry_delay = const.NSX_API_RETRY_DELAY

        for attempt in range(retries):
            response = requests.request(
                method,
                url,
                headers=self.session_headers,
                data=data,
                params=params,
                timeout=self.timeout,
                verify=self.verify_ssl
            )
            logger.debug("API : RESPONSE status=%s, REQUEST method=%s, url=%s,",
                         response.status_code, method, url)

            if response.status_code in allowed_status_codes:
                return response
            if response.status_code == requests.codes.FORBIDDEN:
                logger.error("Failed to %s %s with status: %s and reason: %s",
                             method, url, response.status_code, response.text)
                if self.username is not None and self.password is not None:
                    msg = "Invalid username/password"
                else:
                    msg = response.text
                raise Exception(msg)

            if 500 <= response.status_code < 600:  # Retry on server errors (5xx)
                retries_left = retries - (attempt + 1)
                logger.error("API : method=%s, url=%s failed with 5xx error code. Re-tries left "
                             "- %s", method, url, retries_left)
                time.sleep(retry_delay)
                continue
            else:
                break

        logger.error("Failed to %s %s with status: %s and reason: %s", method, url,
                     response.status_code, response.text)
        if raise_ex:
            raise DetailedHttpError(response)
        return None

    def get(self, endpoint: str, params=None, raise_ex=False):
        return self._rest_call(method='GET',
                               endpoint=endpoint,
                               data=None,
                               allowed_status_codes=self.ALLOWED_GET_STATUS,
                               params=params, raise_ex=raise_ex)

    def put(self, endpoint: str, body=None, params=None):
        if body and self.content_type == "application/json":
            body = json.dumps(body)
        return self._rest_call(method='PUT',
                               endpoint=endpoint,
                               data=body,
                               allowed_status_codes=self.ALLOWED_PUT_STATUS,
                               params=params)

    def get_list_results(self, endpoint: str, params=None):
        """
        Query method for json API get for list (takes care of pagination)
        """
        resp = self.get(endpoint, params)
        response = resp.json() if resp else None
        if not response:
            return []
        results = response.get('results')
        if results is None or not response.get('result_count'):
            return []
        missing = response.get('result_count') - len(results)
        logger.info("URL %s result_count %d cursor %s", endpoint, response['result_count'],
                    str(response.get('cursor')))
        cursor = response.get('cursor', self.NULL_CURSOR_PREFIX)

        op = '&' if urlparse.urlparse(endpoint).query else '?'

        # we will enter the loop if response does not fit into single page
        while missing > 0 and not cursor.startswith(self.NULL_CURSOR_PREFIX):
            resp = self.get(endpoint + op + 'cursor=' + cursor, params)
            response = resp.json() if resp else None
            if not response:
                return results

            cursor = response.get('cursor', self.NULL_CURSOR_PREFIX)
            missing -= len(response['results'])
            results += response['results']

        logger.info("Got %d items from URL %s", len(results), endpoint)
        return results
